Imports System
Imports System.Threading
Imports System.IO
Imports System.Collections

'
' Instances of this class are used to represent sensors that produce
' a stream of measurement values.
'
Public Class StreamingSensor
    Inherits SimpleSensor
    Implements StreamingSensorListener

    Private consumer As StreamReader ' stream for incoming measurement values
    Private listeners As New ArrayList() ' aggregate listeners here
    Private queue As Queue

    ' queue for available measurements
    '
    ' Constructed by a factory method
    ' object that is in the same package as this class and the 
    ' classes that implement its operations.
    '
    ' i - The object that implements the sensor type-specific
    '     operations this object will provide.
    '
    ' r - a stream reader to use for reading the sensor data
    '
    Public Sub New(ByVal i As StreamingSensorImpl, ByVal r As StreamReader)
        MyBase.New(i)
        consumer = r
        queue = New Queue() ' allocate a queue for this instance
        ' start a thread to deliver measurement values
        ThreadPool.QueueUserWorkItem(AddressOf Me.DeliverValues)
    End Sub

    '
    ' Streaming sensors produce a stream of measurement values.  The
    ' stream of values is produced with a frequency no greater than
    ' the given number of times per minute.
    '
    Public WriteOnly Property SamplingFrequency() As Integer
        Set(ByVal value As Integer)
            ' delegate this to the implementation object
            CType(Impl, StreamingSensorImpl).SamplingFrequency = value
        End Set
    End Property

    '
    ' StreamingSensor objects deliver a stream of values to
    ' interested objects by passing each value to the object's
    ' EnQueueMeasurement method.  The delivery of values is done
    ' using its own thread and is asynchronous of everything else.
    '
    ' value - The measurement value being delivered.
    Public Sub EnqueueMeasure(ByVal value As Integer) _
    Implements StreamingSensorListener.EnqueueMeasure
        queue.Enqueue(value)
    End Sub

    '
    ' Remove the current item from the sensor queue
    '
    Public Function DequeueMeasure() As Integer
        Return CInt(queue.Dequeue())
    End Function


    '
    ' Is there an entry available in the sensor queue
    '
    Public Function IsMeasureAvailable() As Boolean
        Return queue.Count > 0
    End Function

    '
    ' This method asynchronously removes measurement values from the pipe
    ' and delivers them to registered listeners.
    '
    Private Sub DeliverValues(ByVal o As Object)
        While True
            Dim s As String = consumer.ReadLine()
            If s Is Nothing Then
                Return
            End If
            Dim value As Integer
            Try
                value = Convert.ToInt32(s)
            Catch e As System.FormatException
                Return ' ignore bad values
            Catch e As System.OverflowException
                Return ' ignore overflow values
            End Try
            EnqueueMeasure(value)
        End While ' while
    End Sub 'DeliverValues

End Class 'StreamingSensor